import datetime

# --- 1. Define the Parser Function ---

def parse_green_button_data(file_path='demofile.txt'):
    """
    Reads the Green Button file and extracts key data from Interval Readings 
    using simple line-by-line string parsing.
    """
    
    # 1. Read the file content and split at 'BREAK' to get the usage data (second part)
    try:
        with open(file_path, 'r', encoding='utf-8') as f:
            full_content = f.read()
    except FileNotFoundError:
        print(f"Error: File not found at {file_path}")
        return []

    # Find the position of the separator 'BREAK'
    if 'BREAK' not in full_content:
        print("Error: Could not find 'BREAK' separator in file.")
        return []
    
    # Get the usage XML content (everything after the BREAK)
    usage_xml_content = full_content.split('BREAK', 1)[1]
    
    # Initialize variables
    all_readings = []
    current_reading = {}
    
    # Flag to track if we are currently inside an IntervalReading block
    in_reading_block = False 

    # --- 2. Process Line-by-Line ---
    
    # Split the usage content into individual lines for simple iteration
    for line in usage_xml_content.splitlines():
        line = line.strip() # Remove leading/trailing spaces
        
        # Check for the start of a new record (IntervalReading)
        if line.startswith('<IntervalReading>'):
            in_reading_block = True
            current_reading = {} # Start a new dictionary for the record
            continue 

        # Check for the end of a record
        if line.startswith('</IntervalReading>'):
            in_reading_block = False
            
            # Ensure we have a complete record before adding it
            if all(key in current_reading for key in ['start', 'value', 'tou']):
                
                # --- Final Data Cleanup and Conversion ---
                
                # The raw value is mWh, convert to kWh by dividing by 1,000,000 
                # (10^3 for mWh to Wh, then 1000 for Wh to kWh)
                raw_value = int(current_reading['value'])
                consumption_kwh = raw_value / 1000000.0 
                
                # Convert Unix timestamp (seconds) to a readable date/time (EST)
                start_timestamp = int(current_reading['start'])
                
                # Unix timestamp is UTC. tzOffset is -18000 seconds (-5 hours) for EST.
                # Subtracting 5 hours (5 * 3600 = 18000 seconds) for EST conversion
                local_timestamp = start_timestamp - 18000 
                
                local_datetime = datetime.datetime.fromtimestamp(local_timestamp)

                # Store the final, cleaned record
                all_readings.append({
                    'Start_Time_EST': local_datetime.strftime('%Y-%m-%d %H:%M'),
                    'Consumption_kWh': round(consumption_kwh, 3), # Round to 3 decimal places
                    'TOU_Code': current_reading['tou']
                })
            
            # Reset for the next record
            current_reading = {} 
            continue

        # --- 3. Extract Key Values if inside a record block ---
        
        if in_reading_block:
            if line.startswith('<start>'):
                # Extract the content between <start> and </start>
                value = line.replace('<start>', '').replace('</start>', '')
                current_reading['start'] = value
            
            elif line.startswith('<value>'):
                # Extract the content between <value> and </value>
                value = line.replace('<value>', '').replace('</value>', '')
                current_reading['value'] = value
            
            elif line.startswith('<tou>'):
                # Extract the content between <tou> and </tou>
                value = line.replace('<tou>', '').replace('</tou>', '')
                current_reading['tou'] = value

    return all_readings

# --- 4. Run the Parser and Print Results ---

if __name__ == '__main__':
    
    data_list = parse_green_button_data()
    
    if not data_list:
        print("No valid energy usage data records were extracted.")
    else:
        print("## Extracted Energy Usage Records (First 5 and Last 5)")
        print("-" * 50)
        
        # Print a header
        print("{:<20} {:<20} {:<10}".format("Start Time (EST)", "Consumption (kWh)", "TOU Code"))
        print("{:<20} {:<20} {:<10}".format("--------------------", "--------------------", "----------"))
        
        # Print the first few records
        for record in data_list[:5]:
            print("{:<20} {:<20} {:<10}".format(
                record['Start_Time_EST'], 
                record['Consumption_kWh'], 
                record['TOU_Code']
            ))

        print("\n... and", len(data_list) - 10, "more records ...\n")
        
        # Print the last few records
        for record in data_list[-5:]:
            print("{:<20} {:<20} {:<10}".format(
                record['Start_Time_EST'], 
                record['Consumption_kWh'], 
                record['TOU_Code']
            ))













import xml.etree.ElementTree as ET

# Define the XML namespaces
NAMESPACES = {
    'espi': 'http://naesb.org/espi'
}

def sum_usage_by_tou(file_path='demofile.txt'):
    """Parses Green Button data and sums energy usage by Time-of-Use code."""
    
    # 1. Load and prepare the usage data (the second XML part)
    try:
        with open(file_path, 'r', encoding='utf-8') as f:
            xml_content = f.read().split('BREAK')[1]
        root = ET.fromstring(xml_content)
    except Exception as e:
        print(f"Error loading/parsing XML usage data: {e}")
        return

    # 2. Get the energy unit conversion factor
    multiplier_elem = root.find(".//espi:ReadingType/espi:powerOfTenMultiplier", NAMESPACES)
    multiplier = int(multiplier_elem.text) if multiplier_elem is not None else -3
    conversion_factor = 10 ** abs(multiplier)
    
    # 3. Initialize the totals dictionary
    tou_totals = {'1': 0, '2': 0, '3': 0}
    
    # 4. Iterate through all IntervalReading elements
    for reading in root.findall(".//espi:IntervalReading", NAMESPACES):
        tou_elem = reading.find("espi:tou", NAMESPACES)
        value_elem = reading.find("espi:value", NAMESPACES)
        
        if tou_elem is not None and value_elem is not None:
            tou_code = tou_elem.text
            raw_value = int(value_elem.text)
            
            # Sum the raw value for the corresponding TOU code
            if tou_code in tou_totals:
                tou_totals[tou_code] += raw_value

    # 5. Output the results
    print("## ✅ Total Energy Consumption by Time-of-Use Period (kWh)")
    print(f"(Raw values are in milliWatt-hours (mWh), converted to Kilowatt-hours (kWh) by dividing by {conversion_factor * 1000:,.0f}.)")
    print("\n| TOU Code | Price Tier | Total Consumption (kWh) |")
    print("| :---: | :--- | :---: |")

    # Convert the summed mWh to kWh (divide by 1,000,000)
    for code, raw_total in tou_totals.items():
        total_kwh = raw_total / (conversion_factor * 1000)
        
        tier_map = {'1': 'On-Peak (HIGH)', '2': 'Mid-Peak (MEDIUM)', '3': 'Off-Peak (LOW)'}
        tier = tier_map.get(code, 'Unknown')
        
        print(f"| {code} | {tier} | {total_kwh:,.2f} kWh |")

if __name__ == '__main__':
    sum_usage_by_tou()