import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.NodeList;
import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;

import java.io.StringReader;
import java.util.ArrayList;
import java.util.List;
import java.util.Date;
import java.text.SimpleDateFormat;

import java.util.HashMap;
import java.util.Map;

/**
 * A simple class to hold one extracted energy reading record.
 * This makes the data structure easy to work with for analysis.
 */
class Record {
    public String startTimeEST;
    public double consumptionKWh;
    public String touCode;

    public Record(String startTimeEST, double consumptionKWh, String touCode) {
        this.startTimeEST = startTimeEST;
        this.consumptionKWh = consumptionKWh;
        this.touCode = touCode;
    }

    @Override
    public String toString() {
        return String.format("%-20s %-20.3f %-10s", startTimeEST, consumptionKWh, touCode);
    }
}

public class XMLReader {

    // --- 1. Main Execution Block ---
    public static void main(String[] args) {
        try {
            // Read the full file content
            String fileContent = new String(java.nio.file.Files.readAllBytes(java.nio.file.Paths.get("demofile.txt")));
            
            // Isolate the usage data (the part after 'BREAK')
            if (!fileContent.contains("BREAK")) {
                System.out.println("Error: Could not find 'BREAK' separator in file.");
                return;
            }
            String usageXml = fileContent.split("BREAK", 2)[1];

            // Parse the data and get the list of records
            List<Record> dataList = parseUsageData(usageXml);

            // Print the results
            if (!dataList.isEmpty()) {
                printResults(dataList);
                
                // Example: Students can add their analysis code here
                aggregateUsage(dataList);
            }
            
        } catch (Exception e) {
            System.out.println("\nAn error occurred during file processing or XML parsing.");
            e.printStackTrace();
        }
    }
    
    // --- 2. The Core Parsing Logic ---
    private static List<Record> parseUsageData(String usageXml) throws Exception {
        List<Record> allReadings = new ArrayList<>();
        
        // Setup the XML parser
        DocumentBuilderFactory dbf = DocumentBuilderFactory.newInstance();
        dbf.setNamespaceAware(true); // Must be true for Green Button XML
        DocumentBuilder db = dbf.newDocumentBuilder();
        
        // Parse the usage XML string
        Document doc = db.parse(new org.xml.sax.InputSource(new StringReader(usageXml)));
        doc.getDocumentElement().normalize();

        // The parser needs the full namespace URI to find the elements
        String espiNamespace = "http://naesb.org/espi";

        // Find all IntervalReading nodes
        NodeList readings = doc.getElementsByTagNameNS(espiNamespace, "IntervalReading");
        
        // Date formatter for output
        SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM-dd HH:mm");

        for (int i = 0; i < readings.getLength(); i++) {
            Element reading = (Element) readings.item(i);
            
            // Extract values using getElementsByTagNameNS on the current element
            // This is simpler than manual string parsing in Java
            String startTag = "start";
            String valueTag = "value";
            String touTag = "tou";
            
            // Get raw data
            long startTimestamp = Long.parseLong(reading.getElementsByTagNameNS(espiNamespace, startTag).item(0).getTextContent());
            long rawValue = Long.parseLong(reading.getElementsByTagNameNS(espiNamespace, valueTag).item(0).getTextContent());
            String touCode = reading.getElementsByTagNameNS(espiNamespace, touTag).item(0).getTextContent();
            
            // --- Conversions for Analysis ---
            
            // 1. Consumption: Raw value (mWh) to kWh (divide by 1,000,000)
            double consumptionKWh = (double) rawValue / 1000000.0;
            
            // 2. Time: Unix timestamp (seconds) to formatted EST time.
            // Subtract 18000 seconds (5 hours) to convert from UTC to EST
            long localTimestampSeconds = startTimestamp - 18000;
            Date startDate = new Date(localTimestampSeconds * 1000); 
            String startTimeEST = sdf.format(startDate);
            
            // Add the final record to the list
            allReadings.add(new Record(startTimeEST, consumptionKWh, touCode));
        }
        
        return allReadings;
    }

    // --- 3. Output Function ---
    private static void printResults(List<Record> dataList) {
        System.out.println("## Extracted Energy Usage Records (First 5 and Last 5)");
        System.out.println("--------------------------------------------------");
        
        // Print a header
        System.out.printf("%-20s %-20s %-10s\n", "Start Time (EST)", "Consumption (kWh)", "TOU Code");
        System.out.printf("%-20s %-20s %-10s\n", "--------------------", "--------------------", "----------");
        
        // Print the first few records
        for (int i = 0; i < Math.min(5, dataList.size()); i++) {
            System.out.println(dataList.get(i));
        }

        if (dataList.size() > 10) {
            System.out.println("\n... and " + (dataList.size() - 10) + " more records ...\n");
            
            // Print the last few records
            for (int i = dataList.size() - 5; i < dataList.size(); i++) {
                System.out.println(dataList.get(i));
            }
        }
    }
    
    // --- 4. Example Analysis Function (for students) ---
    private static void aggregateUsage(List<Record> dataList) {
        Map<String, Double> totalConsumption = new HashMap<>();
        totalConsumption.put("1", 0.0);
        totalConsumption.put("2", 0.0);
        totalConsumption.put("3", 0.0);
        
        for (Record record : dataList) {
            String tou = record.touCode;
            double kwh = record.consumptionKWh;
            
            if (totalConsumption.containsKey(tou)) {
                totalConsumption.put(tou, totalConsumption.get(tou) + kwh);
            }
        }
        
        System.out.println("\n\n## Final Aggregated Results");
        System.out.println("--------------------------------");
        System.out.printf("Total On-Peak (TOU 1):    %,.2f kWh\n", totalConsumption.get("1"));
        System.out.printf("Total Mid-Peak (TOU 2):   %,.2f kWh\n", totalConsumption.get("2"));
        System.out.printf("Total Off-Peak (TOU 3):   %,.2f kWh\n", totalConsumption.get("3"));
    }
}